<?php
// lib.php — common helpers
declare(strict_types=1);
session_start();

// ====== CONFIG ======
const API_TOKEN = '633588354482ef310a571f22'; // Link4m API token
const TOKEN_TTL  = 3600;  // token expires after 60 minutes
const RATE_LIMIT_COUNT = 2;      // 2 times
const RATE_LIMIT_WINDOW = 86400; // per 24h

// Paths
define('BASE_DIR', __DIR__);
define('DATA_DIR', BASE_DIR . '/data');
define('TOKENS_FILE', DATA_DIR . '/tokens.json');
define('IP_FILE', DATA_DIR . '/ip_usage.json');
define('ACCOUNTS_FILE', BASE_DIR . '/accounts/accounts.txt');

// Ensure folders/files
if (!is_dir(DATA_DIR)) { @mkdir(DATA_DIR, 0755, true); }
if (!file_exists(TOKENS_FILE)) { @file_put_contents(TOKENS_FILE, "{}"); }
if (!file_exists(IP_FILE)) { @file_put_contents(IP_FILE, "{}"); }

// ====== UTILS ======
function getClientIp(): string {
    $keys = ['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP','REMOTE_ADDR'];
    foreach ($keys as $k) {
        if (!empty($_SERVER[$k])) {
            $ip = trim(explode(',', $_SERVER[$k])[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
        }
    }
    return '0.0.0.0';
}

function readJson(string $file): array {
    $h = @fopen($file, 'r');
    if (!$h) return [];
    @flock($h, LOCK_SH);
    $raw = stream_get_contents($h);
    @flock($h, LOCK_UN);
    @fclose($h);
    $data = json_decode($raw ?: "[]", true);
    return is_array($data) ? $data : [];
}

function writeJson(string $file, array $data): bool {
    $h = @fopen($file, 'c+');
    if (!$h) return false;
    @flock($h, LOCK_EX);
    ftruncate($h, 0);
    $ok = fwrite($h, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)) !== false;
    fflush($h);
    @flock($h, LOCK_UN);
    @fclose($h);
    return $ok;
}

// Auto-detect subfolder (e.g., /gfn)
function baseUrl(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $script = $_SERVER['SCRIPT_NAME'] ?? '/';
    $dir    = rtrim(str_replace('\\','/', dirname($script)), '/');
    $path   = ($dir === '/' || $dir === '') ? '' : $dir;
    return $scheme . '://' . $host . $path;
}

// Rate limit: returns [ok(bool), remaining(int), reset_ts(int)]
function checkRateLimit(string $ip): array {
    $now = time();
    $map = readJson(IP_FILE);
    $list = $map[$ip] ?? [];
    // keep only within window
    $list = array_values(array_filter($list, fn($ts) => ($now - (int)$ts) < RATE_LIMIT_WINDOW));
    $count = count($list);
    $remaining = max(0, RATE_LIMIT_COUNT - $count);
    $reset_ts = $list ? (min($list) + RATE_LIMIT_WINDOW) : ($now + RATE_LIMIT_WINDOW);
    $map[$ip] = $list;
    writeJson(IP_FILE, $map);
    return [$count < RATE_LIMIT_COUNT, $remaining, $reset_ts];
}

function recordUsage(string $ip): void {
    $map = readJson(IP_FILE);
    $list = $map[$ip] ?? [];
    $list[] = time();
    $map[$ip] = $list;
    writeJson(IP_FILE, $map);
}

function createToken(): string {
    return bin2hex(random_bytes(16));
}

function loadTokens(): array { return readJson(TOKENS_FILE); }
function saveTokens(array $t): bool { return writeJson(TOKENS_FILE, $t); }

function cleanupTokens(array &$tokens): void {
    $now = time();
    foreach ($tokens as $tk => $info) {
        if (!isset($info['created_at']) || ($now - (int)$info['created_at']) > TOKEN_TTL || !isset($info['ip'])) {
            unset($tokens[$tk]);
        }
    }
}

// Link4m shorten
function shortenWithLink4m(string $targetUrl) {
    $api = API_TOKEN;
    $endpoint = "https://link4m.co/api-shorten/v2?api={$api}&url=" . rawurlencode($targetUrl);

    $context = stream_context_create(['http' => ['timeout' => 10]]);
    $resp = @file_get_contents($endpoint, false, $context);
    if ($resp === false) {
        if (function_exists('curl_init')) {
            $ch = curl_init($endpoint);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 10,
                CURLOPT_FOLLOWLOCATION => true,
            ]);
            $resp = curl_exec($ch);
            curl_close($ch);
        }
    }
    if ($resp === false) return false;

    $json = json_decode($resp, true);
    if (isset($json['status']) && $json['status'] === 'success' && !empty($json['shortenedUrl'])) {
        return $json['shortenedUrl'];
    }
    return false;
}

function getRandomAccount(): ?string {
    if (!file_exists(ACCOUNTS_FILE)) return null;
    $lines = @file(ACCOUNTS_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines || !count($lines)) return null;
    $pick = trim($lines[array_rand($lines)]);
    return $pick; // tk:mk
}

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
